using System;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class holds user information returned by remote VistALink server 
	/// (M listener, .NET listener) in response to VistALink get user demographics request. 
	/// </summary>
	public class RemoteUserInfo
	{
		private string _duz, _title, _serviceSection, _language;
		private Timeout	_timeout;

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_USER_INFO_NODE_NAME = "UserInfo";
		private const string XMLCONSTS_DUZ_UI_ATTRIBUTE_NAME = "duz";
		private const string XMLCONSTS_TITLE_UI_ATTRIBUTE_NAME = "title";
		private const string XMLCONSTS_SERVICESECTION_UI_ATTRIBUTE_NAME = "serviceSection";
		private const string XMLCONSTS_LANGUAGE_UI_ATTRIBUTE_NAME = "language";
		private const string XMLCONSTS_TIMEOUT_UI_ATTRIBUTE_NAME = "timeout";

		/// <summary>
		/// This constructor specifies remote user information for VistA systems.
		/// All parameters are required.
		/// </summary>
		/// <param name="duz">User's VistA DUZ</param>
		/// <param name="title">User's title at the site.</param>
		/// <param name="serviceSection">User's service section at the site.</param>
		/// <param name="language">User's preferred language.</param>
		/// <param name="timeout">User's timeout.</param>
		public RemoteUserInfo( string duz, string title, string serviceSection, string language, Timeout timeout )
		{
			if( duz == null )
				throw( new ArgumentNullException( "duz" ) );

			if( title == null )
				throw( new ArgumentNullException( "title" ) );

			if( serviceSection == null )
				throw( new ArgumentNullException( "serviceSection" ) );

			if( language == null )
				throw( new ArgumentNullException( "language" ) );

			if( timeout == null )
				throw( new ArgumentNullException( "timeout" ) );

			_duz = duz;
			_title = title;
			_serviceSection = serviceSection;
			_language = language;
			_timeout = timeout;
		}

		// TODO: Add unit tests. 
		/// <summary>
		/// XML deserialization factory method retrieving a user information XML element 
		/// from a list of descendants of a given node by its unique tag name, parsing it
		/// and returning remote user info object created from it.
		/// </summary>
		/// <param name="rootElement">XML node containing remote user info element among its children.</param>
		/// <returns>Remote user info object deserialized from XML.</returns>
		public static RemoteUserInfo ParseFindChildNodeByUniqueTagName( XmlElement rootElement )
		{
			if( rootElement == null )
				throw( new ArgumentNullException( "rootElement" ) );

			return Parse( XmlUtility.ParseGetRequiredElementByUniqueTagName( rootElement, XMLCONSTS_USER_INFO_NODE_NAME ) );
		}

		/// <summary>
		/// XML deserialization factory method parsing supplied remote user information XML element
		/// and returning remote user info object created from it.
		/// </summary>
		/// <param name="userInfoNode">XML element containing remote user info.</param>
		/// <returns>Remote user info object deserialized from XML.</returns>
		public static RemoteUserInfo Parse( XmlNode userInfoNode )
		{
			if( userInfoNode == null )
				throw( new ArgumentNullException( "userInfoNode" ) );

			XmlElement _userInfoElement = XmlUtility.ParseValidateConvertNodeToElement( userInfoNode, XMLCONSTS_USER_INFO_NODE_NAME );

			return new RemoteUserInfo( 
				XmlUtility.ParseGetRequiredAttributeValue( _userInfoElement, XMLCONSTS_DUZ_UI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _userInfoElement, XMLCONSTS_TITLE_UI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _userInfoElement, XMLCONSTS_SERVICESECTION_UI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _userInfoElement, XMLCONSTS_LANGUAGE_UI_ATTRIBUTE_NAME ),
				Timeout.ParseStringInSeconds( XmlUtility.ParseGetRequiredAttributeValue( _userInfoElement, XMLCONSTS_TIMEOUT_UI_ATTRIBUTE_NAME ) ) );
		}

		/// <summary>
		/// XML serialization method writing out remote user 
		/// info to suppied XML writer.
		/// </summary>
		/// <param name="writer">XmlWriter to write to.</param>
		public void WriteUserInfoToXml( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_USER_INFO_NODE_NAME );

			writer.WriteAttributeString( XMLCONSTS_DUZ_UI_ATTRIBUTE_NAME, _duz );
			writer.WriteAttributeString( XMLCONSTS_TITLE_UI_ATTRIBUTE_NAME, _title );
			writer.WriteAttributeString( XMLCONSTS_SERVICESECTION_UI_ATTRIBUTE_NAME, _serviceSection );
			writer.WriteAttributeString( XMLCONSTS_LANGUAGE_UI_ATTRIBUTE_NAME, _language );
			writer.WriteAttributeString( XMLCONSTS_TIMEOUT_UI_ATTRIBUTE_NAME, _timeout.ToStringSeconds() );

			writer.WriteEndElement();
		}

		/// <summary>
		/// User's VistA DUZ
		/// </summary>
		public string Duz
		{
			get
			{
				return _duz;
			}
		}
		
		/// <summary>
		/// User's title at the site.
		/// </summary>
		public string Title
		{
			get
			{
				return _title;
			}
		}

		/// <summary>
		/// User's service section at the site.
		/// </summary>
		public string ServiceSection
		{
			get
			{
				return _serviceSection;
			}
		}

		/// <summary>
		/// User's preferred language.
		/// </summary>
		public string Language
		{
			get
			{
				return _language;
			}
		}

		/// <summary>
		/// User's timeout.
		/// </summary>
		public Timeout Timeout
		{
			get
			{
				return _timeout;
			}
		}
	}
}
